/*
 * Cheops Network User Interface
 *
 * Copyright (C) 1999, Adtran, Inc.
 * 
 * Distributed under the terms of the GNU GPL
 *
 */

#include <gtk/gtk.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <arpa/inet.h>
#include <string.h>
#include <netdb.h>
#include "cheops.h"


struct widgets {
	int data;
	GtkWidget *one;
	GtkWidget *two;
	GtkWidget *status;
	GtkWidget *toggle;
	GtkWidget *window;
	struct net_page *np;
};

static void do_title(GtkWidget *w);

static void cancel_network(GtkWidget *w)
{
	struct widgets *wids = (struct widgets *)gtk_object_get_user_data(GTK_OBJECT(w));
	gtk_widget_destroy(wids->window);
	if (wids)
		g_free(wids);
}

static void netmask_focus(GtkWidget *w, GtkWidget *w2)
{
	char *c;
	struct hostent *hp;
	struct in_addr ia;
	char hostaddr[80];
	c = gtk_entry_get_text(GTK_ENTRY(w));
	hp = gethostbyname(c);
	if (hp) {
		memcpy(&ia, hp->h_addr, sizeof(ia));
		strncpy(hostaddr, inet_ntoa(ia), sizeof(hostaddr));
		gtk_entry_set_text(GTK_ENTRY(w), hostaddr);
	} else {
		gtk_entry_set_text(GTK_ENTRY(w), "");
	}
	c = gtk_entry_get_text(GTK_ENTRY(w));
	gtk_entry_set_text(GTK_ENTRY(w2), "255.255.255.255");
	if (strlen(c) > 2) {
		if (!strcmp(&c[strlen(c)-2], ".0"))
			gtk_entry_set_text(GTK_ENTRY(w2), "255.255.255.0");
	}
	if (strlen(c) > 4) {
		if (!strcmp(&c[strlen(c)-4], ".0.0"))
			gtk_entry_set_text(GTK_ENTRY(w2), "255.255.0.0");
	}
	if (strlen(c) > 6) {
		if (!strcmp(&c[strlen(c)-2], ".0.0.0"))
			gtk_entry_set_text(GTK_ENTRY(w2), "255.0.0.0");
	}
	
	gtk_widget_grab_focus(w2);
}

static void make_network(GtkWidget *w, GtkWidget *dialog)
{
	struct widgets *wids = (struct widgets *)gtk_object_get_user_data(GTK_OBJECT(w));
	
	char hostaddr[80];
	struct hostent *hp;
	unsigned int nm, addr;
	int res;
	int recursive;
	struct domain *d;
	struct network *n;
	if (!valid_np(wids->np)) {
		g_free(wids);
		gtk_widget_destroy(dialog);
		return;
	}
	if (wids->data == 2) {
		d = wids->np->dcontents;
		while(d) {
			if (!strcasecmp(d->domain, gtk_entry_get_text(GTK_ENTRY(wids->one))))
				break;
			d=d->next;
		}
		if (d) {
			gtk_widget_show(wids->status);
			gtk_statusbar_pop(GTK_STATUSBAR(wids->status), 1);
			gtk_statusbar_push(GTK_STATUSBAR(wids->status), 1, "Already there");
			gtk_entry_select_region(GTK_ENTRY(wids->one), 0, -1);
			return;
		}
		recursive = GTK_TOGGLE_BUTTON(wids->toggle)->active;
		/* Special case: request a domain */
		if (!get_server(gtk_entry_get_text(GTK_ENTRY(wids->one)),1)) {
			gtk_widget_show(wids->status);
			gtk_entry_select_region(GTK_ENTRY(wids->one), 0, -1);
			gtk_statusbar_pop(GTK_STATUSBAR(wids->status), 1);
			gtk_statusbar_push(GTK_STATUSBAR(wids->status), 1, "Invalid Domain");
			return;
		}
		gtk_widget_set_sensitive(wids->window, FALSE);
		res = gather_hosts(wids->np, gtk_entry_get_text(GTK_ENTRY(wids->one)), recursive, 1);
		gtk_widget_set_sensitive(wids->window, TRUE);
		if (res < 1) {
			/* Give it one more try, just in case */
			get_server(gtk_entry_get_text(GTK_ENTRY(wids->one)), 1);
			res = gather_hosts(wids->np, gtk_entry_get_text(GTK_ENTRY(wids->one)), recursive, 0);
		}
			
		if (res < 1) {
			gtk_widget_show(wids->status);
			gtk_statusbar_pop(GTK_STATUSBAR(wids->status), 1);
			gtk_statusbar_push(GTK_STATUSBAR(wids->status), 1, res ? "Invalid Domain" : "No hosts available");
			gtk_entry_select_region(GTK_ENTRY(wids->one), 0, -1);
			return;
		}
	} else {
		/* Network or host */
		
		hp = gethostbyname(gtk_entry_get_text(GTK_ENTRY(wids->one)));
		if (!hp) {
			gtk_widget_show(wids->status);
			gtk_entry_select_region(GTK_ENTRY(wids->one), 0, -1);
			gtk_statusbar_pop(GTK_STATUSBAR(wids->status), 1);
			gtk_statusbar_push(GTK_STATUSBAR(wids->status), 1, "Invalid Host");
			return;
		}
		addr = *(unsigned int *)(hp->h_addr);
		strncpy(hostaddr, inet_ntoa(*(struct in_addr *)hp->h_addr), sizeof(hostaddr));
		if (wids->two) {
			hp = gethostbyname(gtk_entry_get_text(GTK_ENTRY(wids->two)));
			if (!hp) {
				gtk_widget_show(wids->status);
				gtk_entry_select_region(GTK_ENTRY(wids->two), 0, -1);
				gtk_statusbar_pop(GTK_STATUSBAR(wids->status), 1);
				gtk_statusbar_push(GTK_STATUSBAR(wids->status), 1, "Invalid Netmask");
				return;
			}
			nm = htonl(*(unsigned int *)hp->h_addr);
			while(nm & 0x80000000)
				nm = (nm << 1) & 0xFFFFFFFE;
			if (nm) {
				gtk_widget_show(wids->status);
				gtk_entry_select_region(GTK_ENTRY(wids->two), 0, -1);
				gtk_statusbar_pop(GTK_STATUSBAR(wids->status), 1);
				gtk_statusbar_push(GTK_STATUSBAR(wids->status), 1, "Invalid Netmask");
				return;
			}
			nm = htonl(*(unsigned int *)hp->h_addr);
		} else 
			nm = 0xFFFFFFFF;
		n  = wids->np->ncontents;
		while(n) {
			if ((nm == ntohl(n->net_mask)) && (addr == n->net_addr))
				break;
			n=n->next;
		}
		if (n) {
			gtk_widget_show(wids->status);
			gtk_entry_select_region(GTK_ENTRY(wids->one), 0, -1);
			gtk_statusbar_pop(GTK_STATUSBAR(wids->status), 1);
			gtk_statusbar_push(GTK_STATUSBAR(wids->status), 1, "Already there");
			return;
		}
	
		if (wids->two)
			discover_network_a(wids->np, hostaddr,
					 	gtk_entry_get_text(GTK_ENTRY(wids->two)), 1); 
		else
			discover_network_a(wids->np, hostaddr,
						"255.255.255.255", 1);
	}
	g_free(wids);
	gtk_widget_destroy(dialog);
}

void do_network(GtkWidget *ign, int data)
{
	GtkWidget *dialog;
	GtkWidget *ok;
	GtkWidget *cancel;
	GtkWidget *label;
	GtkWidget *label2 = NULL;
	GtkWidget *hbox;
	GtkWidget *entry;
	GtkWidget *status;
	GtkWidget *netmask = NULL;
	GtkWidget *toggle = NULL;
	struct widgets *w;
	

	if (!GTK_NOTEBOOK(main_window.notebook)->children)
		return;
	
	w = g_new0(struct widgets, 1);
	w->window = dialog = gtk_dialog_new();
	hbox = gtk_hbox_new(FALSE, 5);
	w->one = entry = gtk_entry_new();
	w->np = current_page;
	if (data)
		w->two = netmask = gtk_entry_new();
	else
		w->two = NULL;
	if (data == 1)
		label = gtk_label_new("Network address");
	else if (data == 2)
		label = gtk_label_new("Domain name");
	else
		label = gtk_label_new("Host name/address");
	if (data == 1)
		label2 = gtk_label_new("Netmask");
	cancel = gtk_button_new_with_label("Cancel");
	ok = gtk_button_new_with_label("OK");
	w->status = status = gtk_statusbar_new();
	if (data == 2) 
		w->toggle = toggle = gtk_check_button_new_with_label("Recurse Subdomains");
	w->data = data;
	if (data == 1)
		gtk_window_set_title(GTK_WINDOW(dialog), "Add Network");
	else if (data == 2)
		gtk_window_set_title(GTK_WINDOW(dialog), "Add Domain");
	else
		gtk_window_set_title(GTK_WINDOW(dialog), "Add Host");
		
	gtk_object_set_user_data(GTK_OBJECT(ok), w);
	gtk_object_set_user_data(GTK_OBJECT(cancel), w);
	gtk_object_set_user_data(GTK_OBJECT(entry), w);
	if (data == 1)
		gtk_object_set_user_data(GTK_OBJECT(netmask), w);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(hbox), entry, FALSE, FALSE, 5);
	if (data == 1) {
		gtk_box_pack_start(GTK_BOX(hbox), label2, FALSE, FALSE, 5);
		gtk_box_pack_start(GTK_BOX(hbox), netmask, FALSE, FALSE, 5);
	}

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, TRUE,
                           TRUE, 0);
	if (data == 2)
		gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), toggle, TRUE,
                           TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), status, TRUE,
                           TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), ok, TRUE,
                           TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), cancel, TRUE,
                           TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(ok), "clicked", GTK_SIGNAL_FUNC(make_network), dialog);
	if (data == 1) {
		gtk_signal_connect(GTK_OBJECT(entry), "activate", GTK_SIGNAL_FUNC(netmask_focus), netmask);
#if 0
		gtk_signal_connect(GTK_OBJECT(entry), "leave_notify_event", GTK_SIGNAL_FUNC(netmask_focus), netmask);
#endif
	} else
		gtk_signal_connect(GTK_OBJECT(entry), "activate", GTK_SIGNAL_FUNC(make_network), dialog);
	if (data == 1)
		gtk_signal_connect(GTK_OBJECT(netmask), "activate", GTK_SIGNAL_FUNC(make_network), dialog);
	gtk_signal_connect(GTK_OBJECT(cancel), "clicked", GTK_SIGNAL_FUNC(cancel_network), dialog);
	gtk_widget_show(hbox);
	gtk_widget_show(entry);
	if (data == 1) {
		gtk_widget_show(label2);
		gtk_widget_show(netmask);
	} else if (data == 2) 
		gtk_widget_show(toggle);
	
	gtk_widget_show(label);
	gtk_widget_show(cancel);
	gtk_widget_show(ok);
	gtk_widget_grab_focus(entry);
	gtk_widget_show(dialog);
}

static int destroy_hand(void *data)
{
	struct net_page *np = (struct net_page *)gtk_object_get_user_data(GTK_OBJECT(data));
	np->lasthand = NULL;
	np->lastid=0;
#ifdef USE_CANVAS
	gtk_object_destroy(GTK_OBJECT(data));
#else
	gtk_widget_destroy(GTK_WIDGET(data));
#endif
	return FALSE;
}

static void add_hand(struct net_page *np, int x, int y, char *filename)
{
#ifdef USE_CANVAS
	GnomeCanvasItem *pixmap;
	GdkImlibImage *im;
#else
	GtkWidget *pixmap;
	GtkWidget *fixed = np->fixed;
	struct pixmap_cache *tmp;
#endif
	GtkWidget *hbar = GTK_SCROLLED_WINDOW(np->pane)->hscrollbar;
	GtkWidget *vbar = GTK_SCROLLED_WINDOW(np->pane)->vscrollbar;
	
	if (np->lastid) {
		gtk_timeout_remove(np->lastid);
		if (np->lasthand)
#ifdef USE_CANVAS
			gtk_object_destroy(GTK_OBJECT(np->lasthand));
#else
			gtk_widget_destroy(np->lasthand);
#endif
		np->lastid = 0;
	}
	
#ifdef USE_CANVAS
	im = gdk_imlib_load_image(find_file(filename));
	pixmap = gnome_canvas_item_new(gnome_canvas_root(GNOME_CANVAS(np->canvas)),
				     GNOME_TYPE_CANVAS_IMAGE,
				     "x", (double)x,
				     "y", (double)y,
				     "image", im, 
				     "width", (double)im->rgb_width,
				     "height", (double)im->rgb_height,
				     NULL);
	gnome_canvas_item_raise_to_top(pixmap);
#else
	tmp = get_pixmap(filename);
	pixmap = gtk_pixmap_new (tmp->pixmap, tmp->mask);
	gtk_widget_show(pixmap);
	gtk_fixed_put(GTK_FIXED(fixed), pixmap, x + 20, y + np->icon_height - 30);
#endif
	gtk_object_set_user_data(GTK_OBJECT(pixmap), np);
	np->lastid = gtk_timeout_add(6000, destroy_hand, pixmap);
	np->lasthand = pixmap;
	x = x - np->pane->allocation.width/2;
	y = y - np->pane->allocation.height/2;
	if (x < 0)
		x=0;
	if (y < 0)
		y=0;
	gtk_adjustment_set_value(GTK_RANGE(hbar)->adjustment, x);
	gtk_adjustment_set_value(GTK_RANGE(vbar)->adjustment, y);
#ifdef DEBUG
	printf("max is %f, min is %f\n", GTK_RANGE(vbar)->adjustment->upper, GTK_RANGE(vbar)->adjustment->lower);
	printf("max is %f, min is %f\n", GTK_RANGE(hbar)->adjustment->upper, GTK_RANGE(hbar)->adjustment->lower);
#endif

}

static int search(struct net_object *no, char *srch, int exact)
{
	char *str;
	struct in_addr ia;
	int res=0;
#ifdef USE_CANVAS	
	str = ((GnomeCanvasText *)(no->label))->text;
#else
	gtk_label_get(GTK_LABEL(no->label), &str);
#endif
	if (exact) {
		if (!strcasecmp(str, srch))
			res=1;
	} else {
		if (strstr(str, srch))
			res=1;
	}
	if (exact) {
		if (!strcasecmp(no->hostname, srch))
			res=1;
	} else {
		if (strstr(no->hostname, srch))
			res=1;
	}
	ia.s_addr = no->ip_addr;
	if (exact) {
		if (!strcasecmp(inet_ntoa(ia), srch))
			res=1;;
	} else {
		if (strstr(inet_ntoa(ia), srch))
			res=1;
	}
	return res;
}

static void make_search(GtkWidget *w, GtkWidget *dialog)
{
	struct widgets *wids = (struct widgets *)gtk_object_get_user_data(GTK_OBJECT(w));
	
	int exact;
	char *srch;
	struct net_object *no;
#ifdef USE_CANVAS
	double x1, y1, x2, y2;
#endif
	int x;
	exact = GTK_TOGGLE_BUTTON(wids->toggle)->active;
	srch = gtk_entry_get_text(GTK_ENTRY(wids->one));
	if (!strlen(srch))
		return;
#ifdef DEBUG
	printf("Searching%s for %s\n",exact ? " exactly" : "", srch);
#endif
	if (gtk_notebook_current_page(GTK_NOTEBOOK(current_page->notebook))) {
		no = (struct net_object *)
			gtk_clist_get_row_data(GTK_CLIST(current_page->list), 0);
	} else {
		no = current_page->objs;
	}
	if (gtk_notebook_current_page(GTK_NOTEBOOK(current_page->notebook))) {
		for(x=no->id; x < GTK_CLIST(current_page->list)->rows;x++) {
			no = (struct net_object *)
				gtk_clist_get_row_data(GTK_CLIST(current_page->list), x);
			if (search(no, srch, 0))
				break;
		}
	} else {
		while(no) {
			if (search(no, srch, 0))
				break;
			no=no->next;
		}
	}
	current_page->last = no;
	strncpy(current_page->laststr, srch, sizeof(current_page->laststr));
	if (no) {
		char buf[256];
		g_snprintf(buf, sizeof(buf), "Found '%s'",no->hostname);
		set_status(buf);
		if (gtk_notebook_current_page(GTK_NOTEBOOK(current_page->notebook))) {
			gtk_clist_select_row(GTK_CLIST(current_page->list), no->id, -1);
			gtk_clist_moveto(GTK_CLIST(current_page->list), no->id, 0, 0.5, 0.5);
		} else 
#ifdef USE_CANVAS
		{
		       	gnome_canvas_item_get_bounds(no->icon, &x1, &y1, &x2, &y2);
			add_hand(current_page, x1 + current_page->icon_width/2, y1 + current_page->icon_height/2, "point.xpm");
		}
#else
			add_hand(current_page, no->eventbox->allocation.x, no->eventbox->allocation.y, "point.xpm");
#endif
		g_free(wids);
		gtk_widget_destroy(dialog);
	} else {
		gtk_widget_show(wids->status);
		gtk_statusbar_pop(GTK_STATUSBAR(wids->status), 1);
		gtk_statusbar_push(GTK_STATUSBAR(wids->status), 1, "Host not found");
		gtk_entry_select_region(GTK_ENTRY(wids->one), 0, -1);
		return;
	}
}

void do_search()
{
	GtkWidget *dialog;
	GtkWidget *ok;
	GtkWidget *cancel;
	GtkWidget *label;
	GtkWidget *hbox;
	GtkWidget *entry;
	GtkWidget *status;
	GtkWidget *toggle;
	struct widgets *w;
	

	if (!GTK_NOTEBOOK(main_window.notebook)->children)
		return;
	
	w = g_new0(struct widgets, 1);
	w->window = dialog = gtk_dialog_new();
	hbox = gtk_hbox_new(FALSE, 5);
	w->one = entry = gtk_entry_new();
	label = gtk_label_new("Find Host");
	cancel = gtk_button_new_with_label("Cancel");
	ok = gtk_button_new_with_label("OK");
	w->status = status = gtk_statusbar_new();
	w->toggle = toggle = gtk_check_button_new_with_label("Exact Match");
	gtk_window_set_title(GTK_WINDOW(dialog), "Find Host");
		
	gtk_object_set_user_data(GTK_OBJECT(ok), w);
	gtk_object_set_user_data(GTK_OBJECT(cancel), w);
	gtk_object_set_user_data(GTK_OBJECT(entry), w);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(hbox), entry, FALSE, FALSE, 5);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, TRUE,
                           TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), toggle, TRUE,
                           TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), status, TRUE,
                           TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), ok, TRUE,
                           TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), cancel, TRUE,
                           TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(ok), "clicked", GTK_SIGNAL_FUNC(make_search), dialog);
	gtk_signal_connect(GTK_OBJECT(entry), "activate", GTK_SIGNAL_FUNC(make_search), dialog);
	gtk_signal_connect(GTK_OBJECT(cancel), "clicked", GTK_SIGNAL_FUNC(cancel_network), dialog);
	gtk_widget_show(hbox);
	gtk_widget_show(entry);
	gtk_widget_show(toggle);	
	gtk_widget_show(label);
	gtk_widget_show(cancel);
	gtk_widget_show(ok);
	gtk_widget_grab_focus(entry);
	gtk_widget_show(dialog);
}


void search_again()
{
	char *srch;
	int first;
	struct net_object *no;
	int x=0;
#ifdef USE_CANVAS
	double x1, y1, x2, y2;
#endif
	if (!current_page)	
		return;
	srch = current_page->laststr;
	if (!strlen(srch)) {
		set_status("No previous search");
		return;
	}
	if (current_page->last) {
		if (gtk_notebook_current_page(GTK_NOTEBOOK(current_page->notebook))) {
			no = (struct net_object *)
				gtk_clist_get_row_data(GTK_CLIST(current_page->list), 
					current_page->last->id + 1);
			first = 0;
		} else {
			no = current_page->last->next;
			first = 0;
		}
		first=0;
	} else {
		if (gtk_notebook_current_page(GTK_NOTEBOOK(current_page->notebook))) {
			no = (struct net_object *)
				gtk_clist_get_row_data(GTK_CLIST(current_page->list), 0);
			first = 1;
		} else {
			no = current_page->objs;
			first = 1;
		}
	}
	if (gtk_notebook_current_page(GTK_NOTEBOOK(current_page->notebook)) && no) {
		for(x=no->id; x < GTK_CLIST(current_page->list)->rows;x++) {
			no = (struct net_object *)
				gtk_clist_get_row_data(GTK_CLIST(current_page->list), x);
			if (search(no, srch, 0))
				break;
		}
		if ( x >= GTK_CLIST(current_page->list)->rows)
			no=NULL;
	} else {
		while(no) {
			if (search(no, srch, 0))
				break;
			no=no->next;
		}
	}
	current_page->last = no;
	if (no) {
		char buf[256];
		g_snprintf(buf, sizeof(buf), "Found '%s'",no->hostname);
		set_status(buf);
		if (gtk_notebook_current_page(GTK_NOTEBOOK(current_page->notebook))) {
			gtk_clist_select_row(GTK_CLIST(current_page->list), no->id, -1);
			gtk_clist_moveto(GTK_CLIST(current_page->list), no->id, 0, 0.5, 0.5);
		} else
#ifdef USE_CANVAS
		{
		       	gnome_canvas_item_get_bounds(no->icon, &x1, &y1, &x2, &y2);
			add_hand(current_page, x1+current_page->icon_width/2, y1+current_page->icon_height/2 , "point.xpm");
		}
#else
			add_hand(current_page, no->eventbox->allocation.x, no->eventbox->allocation.y, "point.xpm");
#endif
	} else {
		set_status(first ? "No matching hosts" : "No more matching hosts");
	}
}

void delete_edit(GtkWidget *w)
{
	struct pageedit *pe;
	pe = (struct pageedit *)gtk_object_get_user_data(GTK_OBJECT(w));
	if (!pe)
		return;
	if (valid_np(pe->np)) {
		do_title(w);
		pe->np->pe = NULL;
	}
	gtk_widget_destroy(pe->window);
	g_free(pe);
}

int append_network(struct pageedit *pe, struct network *n)
{

	char *ptrs[2];
	char host[256];
	char netmask[256];
	struct in_addr ia;
	int p;
	
	if (!pe)
		return -1;
	ptrs[0] = host;
	ptrs[1] = netmask;
	ia.s_addr = n->net_mask;
	strncpy(netmask, inet_ntoa(ia), sizeof(netmask));
	ia.s_addr = n->net_addr;
	strncpy(host, inet_ntoa(ia), sizeof(host));
	if (!strcmp(netmask, "255.255.255.255") && option_reverse_dns) {
		struct hostent *hp;
		hp = gethostbyaddr((char *)&ia, sizeof(ia), AF_INET);
		if (hp) 
			strncpy(host, hp->h_name, sizeof(host));
	}
	p= gtk_clist_append(GTK_CLIST(pe->networks), ptrs);
	gtk_clist_set_row_data(GTK_CLIST(pe->networks), p, n);
	return p;
}

int append_domain(struct pageedit *pe, struct domain *d)
{
	char *ptrs[2];
	int p;
	if (!pe)
		return -1;
	ptrs[0]=d->domain;
	p=gtk_clist_append(GTK_CLIST(pe->domains), ptrs);
	gtk_clist_set_row_data(GTK_CLIST(pe->domains), p, d);
	return p;
}

static void populate_networks(struct net_page *np, struct pageedit *pe)
{
	int p;
	struct network *n;
	
	n = np->ncontents;
	while(n) {
		p = append_network(pe, n);
		n = n->next;
	}
}

static void populate_domains(struct net_page *np, struct pageedit *pe)
{
	int p;
	struct domain *d;
	d = np->dcontents;
	while(d) {
		p = append_domain(pe, d);
		d = d->next;
	}
}

static void do_network_wrapper(GtkWidget *w, int d)
{
	struct pageedit *pe;
	struct net_page *np;
	pe = (struct pageedit *)gtk_object_get_user_data(GTK_OBJECT(w));
	/* Temporarily munge current_page */
	if (!valid_np(pe->np))
		return;
	np = current_page;
	current_page = pe->np;
	do_network(w, d);
	current_page = np;
}

static void remove_network(GtkWidget *w)
{
	struct pageedit *pe;
	struct network *n;
	struct network *tn, *last;
	int r;
	GList *l;

	pe = (struct pageedit *)gtk_object_get_user_data(GTK_OBJECT(w));
	l = g_list_first(GTK_CLIST(pe->networks)->selection);
	if (!l)
		return;
	if (!valid_np(pe->np))
		return;
	r = GPOINTER_TO_INT(l->data);
	if (r < 0)
		return; 
	n = (struct network *)gtk_clist_get_row_data(GTK_CLIST(pe->networks), r);
	tn = pe->np->ncontents;
	last=NULL;
	while(tn) {
		if (tn == n) {
			if (last) {
				last->next = tn->next;
			} else {
				pe->np->ncontents=tn->next;
			}
			g_free(tn);
			break;
		}
		last=tn;
		tn=tn->next;
	}
	gtk_clist_remove(GTK_CLIST(pe->networks), r);
}

static void remove_domain(GtkWidget *w)
{
	struct pageedit *pe;
	struct domain *d;
	int r;
	GList *l;
	struct domain *td, *last;

	pe = (struct pageedit *)gtk_object_get_user_data(GTK_OBJECT(w));
	l = g_list_first(GTK_CLIST(pe->domains)->selection);
	if (!l)
		return;
	if (!valid_np(pe->np))
		return;
	r = GPOINTER_TO_INT(l->data);
	if (r < 0)
		return; 
	d = (struct domain *)gtk_clist_get_row_data(GTK_CLIST(pe->domains), r);
	td = pe->np->dcontents;
	last=NULL;
	while(td) {
		if (td == d) {
			if (last) {
				last->next = td->next;
			} else {
				pe->np->dcontents=td->next;
			}
			g_free(td);
			break;
		}
		last=td;
		td = td->next;
	}
	gtk_clist_remove(GTK_CLIST(pe->domains), r);
}

static void do_title(GtkWidget *w)
{
	struct pageedit *pe;
	char buf[256], *s;
	pe = (struct pageedit *)gtk_object_get_user_data(GTK_OBJECT(w));
	if (!valid_np(pe->np)) 
		return;
	gtk_label_get(GTK_LABEL(pe->np->label), &s);
	if (strcmp(gtk_entry_get_text(GTK_ENTRY(pe->entry)), s)) {
		gtk_label_set(GTK_LABEL(pe->np->label), 
			      gtk_entry_get_text(GTK_ENTRY(pe->entry)));
		g_snprintf(buf, sizeof(buf), "Page '%s'",
			 gtk_entry_get_text(GTK_ENTRY(pe->entry)));
		gtk_window_set_title(GTK_WINDOW(pe->window), buf);
	}
}

static void do_update(GtkWidget *w)
{
	struct pageedit *pe;
	struct net_page *np;
	do_title(w);
	pe = (struct pageedit *)gtk_object_get_user_data(GTK_OBJECT(w));
	if (!valid_np(pe->np)) 
		return;
	np = current_page;
	current_page = pe->np;
	reload_page();
	current_page = np;
}

void build_edit(struct net_page *np, GtkWidget *cont)
{
	struct pageedit *pe;
	GtkWidget *bbox;
	GtkWidget *bbox2;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *close=NULL;
	GtkWidget *add3;
	GtkWidget *add2;
	GtkWidget *remove2;
	GtkWidget *add;
	GtkWidget *remove;
	GtkWidget *update;
#if (GTK_MINOR_VERSION > 1) || ((GTK_MICRO_VERSION > 4)	&& (GTK_MINOR_VERSION > 0))
	GtkWidget *sw1;
	GtkWidget *sw2;
#endif
	char buf[256], *s;
	static char *titles_net[] = { "Network", "Netmask", NULL };
	static char *titles_dom[] = { "Domain", NULL };
	if (np->pe) {
		gtk_widget_show(np->pe->window);
		return;
	}
	gtk_label_get(GTK_LABEL(np->label), &s);
	g_snprintf(buf, sizeof(buf), "Page '%s'", s);
	np->pe = pe = g_new0(struct pageedit, 1);
	pe->np = np;
	if (cont) {
		pe->window = cont;
	} else {
		pe->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_widget_realize(pe->window);
		fix_icon(pe->window->window);
	}
	pe->networks = gtk_clist_new_with_titles(2, titles_net);
	
#if ((GTK_MINOR_VERSION > 0) && (GTK_MICRO_VERSION > 2))  || (GTK_MAJOR_VERSION > 1)
	gtk_clist_set_column_auto_resize(GTK_CLIST(pe->networks), 0, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(pe->networks), 1, TRUE);
#else
	gtk_clist_set_column_width(GTK_CLIST(pe->networks), 0, 100);
	gtk_clist_set_column_width(GTK_CLIST(pe->networks), 1, 100);
#endif
	pe->domains = gtk_clist_new_with_titles(1, titles_dom);
	vbox = gtk_vbox_new(FALSE, 5);
	bbox = gtk_hbox_new(FALSE, 5);
	bbox2 = gtk_hbox_new(FALSE, 5);
	if (!cont) {
		close = gtk_button_new_with_label("Close");
	}
	update = gtk_button_new_with_label("Update");
	add = gtk_button_new_with_label("Add domain");
	add2 = gtk_button_new_with_label("Add network");
	add3 = gtk_button_new_with_label("Add host");
	remove = gtk_button_new_with_label("Remove domain");
	remove2 = gtk_button_new_with_label("Remove host/network");
	
	pe->entry = gtk_entry_new();
	label = gtk_label_new("Name:");
	hbox = gtk_hbox_new(FALSE, 5);

	gtk_entry_set_text(GTK_ENTRY(pe->entry), s);
	gtk_object_set_user_data(GTK_OBJECT(pe->entry), pe);
	gtk_signal_connect(GTK_OBJECT(pe->entry), "activate", GTK_SIGNAL_FUNC(do_title), NULL);

	gtk_widget_show(pe->entry);
	gtk_widget_show(hbox);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(hbox), pe->entry, TRUE, TRUE, 5);
	
	gtk_box_pack_end(GTK_BOX(bbox2), remove2, FALSE, FALSE, 5);
	gtk_box_pack_end(GTK_BOX(bbox2), add2, FALSE, FALSE, 5);
	gtk_box_pack_end(GTK_BOX(bbox2), add3, FALSE, FALSE, 5);

	if (!cont)
		gtk_box_pack_end(GTK_BOX(bbox), close, FALSE, FALSE, 5);
	gtk_box_pack_end(GTK_BOX(bbox), update, FALSE, FALSE, 5);
	gtk_box_pack_end(GTK_BOX(bbox), remove, FALSE, FALSE, 5);
	gtk_box_pack_end(GTK_BOX(bbox), add, FALSE, FALSE, 5);

	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);

#if (GTK_MINOR_VERSION > 1) || ((GTK_MICRO_VERSION > 4)	&& (GTK_MINOR_VERSION > 0))
	sw1 = gtk_scrolled_window_new(NULL, NULL);
	sw2 = gtk_scrolled_window_new(NULL, NULL);
	gtk_widget_show(sw1);
	gtk_widget_show(sw2);
	gtk_widget_set_usize(sw1, 0, 100);
	gtk_widget_set_usize(sw2, 0, 100);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw1), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw2), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_container_add(GTK_CONTAINER(sw1), pe->networks);
	gtk_container_add(GTK_CONTAINER(sw2), pe->domains);

	gtk_box_pack_start(GTK_BOX(vbox), sw1, TRUE, TRUE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), bbox2, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), sw2, TRUE, TRUE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 5);

#else
	gtk_widget_set_usize(pe->networks, 0, 100);
	gtk_widget_set_usize(pe->domains, 0, 100);
	gtk_clist_set_policy(GTK_CLIST(pe->networks), GTK_POLICY_ALWAYS, GTK_POLICY_AUTOMATIC);
	gtk_clist_set_policy(GTK_CLIST(pe->domains), GTK_POLICY_ALWAYS, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), pe->networks, TRUE, TRUE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), bbox2, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), pe->domains, TRUE, TRUE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 5);
#endif
	
	gtk_container_add(GTK_CONTAINER(pe->window), vbox);
	
	gtk_widget_show(vbox);
	gtk_widget_show(bbox);
	gtk_widget_show(bbox2);
	gtk_widget_show(update);
	gtk_widget_show(pe->networks);
	gtk_widget_show(pe->domains);
	
	populate_networks(np, pe);
	populate_domains(np, pe);
	
	gtk_object_set_user_data(GTK_OBJECT(update), pe);
	gtk_signal_connect(GTK_OBJECT(update), "clicked", GTK_SIGNAL_FUNC(do_update), NULL);
	gtk_object_set_user_data(GTK_OBJECT(add), pe);
	gtk_signal_connect(GTK_OBJECT(add), "clicked", GTK_SIGNAL_FUNC(do_network_wrapper), (void *)2);
	gtk_object_set_user_data(GTK_OBJECT(add2), pe);
	gtk_signal_connect(GTK_OBJECT(add2), "clicked", GTK_SIGNAL_FUNC(do_network_wrapper), (void *)1);
	gtk_object_set_user_data(GTK_OBJECT(add3), pe);
	gtk_signal_connect(GTK_OBJECT(add3), "clicked", GTK_SIGNAL_FUNC(do_network_wrapper), (void*)0);

	gtk_object_set_user_data(GTK_OBJECT(remove2), pe);
	gtk_signal_connect(GTK_OBJECT(remove2), "clicked", GTK_SIGNAL_FUNC(remove_network), NULL);
	gtk_object_set_user_data(GTK_OBJECT(remove), pe);
	gtk_signal_connect(GTK_OBJECT(remove), "clicked", GTK_SIGNAL_FUNC(remove_domain), NULL);

	if (!cont)
		gtk_widget_show(close);
	gtk_widget_show(add);
	gtk_widget_show(remove);
	gtk_widget_show(add2);
	gtk_widget_show(add3);
	gtk_widget_show(remove2);

	gtk_object_set_user_data(GTK_OBJECT(add), pe);
	gtk_object_set_user_data(GTK_OBJECT(add2), pe);
	gtk_object_set_user_data(GTK_OBJECT(add3), pe);
	gtk_object_set_user_data(GTK_OBJECT(remove), pe);
	gtk_object_set_user_data(GTK_OBJECT(remove2), pe);
	gtk_object_set_user_data(GTK_OBJECT(pe->window), pe);
	if (!cont) {
		gtk_object_set_user_data(GTK_OBJECT(close), pe);
		gtk_signal_connect(GTK_OBJECT(close), "clicked", GTK_SIGNAL_FUNC(delete_edit), pe);
		gtk_signal_connect(GTK_OBJECT(pe->window), "delete_event", GTK_SIGNAL_FUNC(delete_edit), pe);
		gtk_widget_grab_focus(close);
		gtk_window_set_title(GTK_WINDOW(pe->window),  buf);
		gtk_widget_show(pe->window);
	}
}

void show_edit(struct net_page *np)
{
	build_edit(np, NULL);
}

